<?php
session_start();

// Pfade zu den JSON-Dateien
$config_file = 'config.json';
$users_file = 'users.json';
$categories_file = 'categories.json';
$links_file = 'links.json';
$texts_file = 'texts.json';
$upload_dir = 'uploads/';

// Sicherstellen, dass das Upload-Verzeichnis existiert
if (!is_dir($upload_dir)) {
    mkdir($upload_dir, 0777, true);
}

// Funktionen zum Laden und Speichern von JSON-Daten
function load_json($file) {
    if (file_exists($file)) {
        return json_decode(file_get_contents($file), true);
    }
    return [];
}

function save_json($file, $data) {
    return file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

// Konfiguration laden
$config = load_json($config_file);

// Prüfen, ob das Setup abgeschlossen ist
if (empty($config) || !isset($config['admin_username'])) {
    header('Location: setup.php');
    exit;
}

// Login-Logik
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
        $username = $_POST['username'] ?? '';
        $password = $_POST['password'] ?? '';

        if ($username === $config['admin_username'] && password_verify($password, $config['admin_password'])) {
            $_SESSION['logged_in'] = true;
            header('Location: admin.php');
            exit;
        } else {
            $login_message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Ungültiger Benutzername oder Passwort.</div>';
        }
    }
    // Zeige Login-Formular
    ?>
    <!DOCTYPE html>
    <html lang="de">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Admin Login - DigiLinks</title>
        <script src="https://cdn.tailwindcss.com"></script>
        <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
        <style>
            body {
                font-family: 'Inter', sans-serif;
                background-color: #f0f2f5;
                display: flex;
                justify-content: center;
                align-items: center;
                min-height: 100vh;
            }
            .login-container {
                background-color: #ffffff;
                padding: 2.5rem;
                border-radius: 0.75rem;
                box-shadow: 0 10px 15px rgba(0, 0, 0, 0.1);
                width: 100%;
                max-width: 400px;
                text-align: center;
            }
            .form-group label {
                display: block;
                text-align: left;
                margin-bottom: 0.5rem;
                font-weight: 600;
                color: #4a5568;
            }
            .form-group input {
                width: 100%;
                padding: 0.75rem;
                border: 1px solid #cbd5e0;
                border-radius: 0.375rem;
                margin-bottom: 1rem;
                box-sizing: border-box;
            }
            .form-group button {
                width: 100%;
                padding: 0.75rem;
                background-color: #4c51bf;
                color: white;
                border: none;
                border-radius: 0.375rem;
                font-size: 1.125rem;
                cursor: pointer;
                transition: background-color 0.3s ease;
            }
            .form-group button:hover {
                background-color: #434190;
            }
        </style>
    </head>
    <body>
        <div class="login-container">
            <h1 class="text-3xl font-bold text-gray-800 mb-4">Admin Login</h1>
            <?php echo $login_message ?? ''; ?>
            <form method="POST">
                <div class="form-group">
                    <label for="username">Benutzername:</label>
                    <input type="text" id="username" name="username" required>
                </div>
                <div class="form-group">
                    <label for="password">Passwort:</label>
                    <input type="password" id="password" name="password" required>
                </div>
                <div class="form-group">
                    <button type="submit" name="login">Anmelden</button>
                </div>
            </form>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Logout-Logik
if (isset($_GET['action']) && $_GET['action'] === 'logout') {
    session_destroy();
    header('Location: admin.php');
    exit;
}

$message = '';
$current_tab = $_GET['tab'] ?? 'general'; // Standard-Tab

// --- Allgemeine Einstellungen (General Settings) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_general_settings'])) {
    $config['linktree_name'] = $_POST['linktree_name'] ?? $config['linktree_name'];
    $config['linktree_icon'] = $_POST['linktree_icon'] ?? ''; // Neues Feld für Linktree Icon
    $config['colors']['text'] = $_POST['text_color'] ?? $config['colors']['text'];
    $config['colors']['page_bg_type'] = $_POST['page_bg_type'] ?? $config['colors']['page_bg_type'];
    $config['colors']['page_bg_value'] = $_POST['page_bg_value'] ?? $config['colors']['page_bg_value'];
    $config['colors']['category_bg_type'] = $_POST['category_bg_type'] ?? $config['colors']['category_bg_type'];
    $config['colors']['category_bg_value'] = $_POST['category_bg_value'] ?? $config['colors']['category_bg_value'];
    $config['colors']['linktree_bg_type'] = $_POST['linktree_bg_type'] ?? $config['colors']['linktree_bg_type'];
    $config['colors']['linktree_bg_value'] = $_POST['linktree_bg_value'] ?? $config['colors']['linktree_bg_value'];
    $config['category_display'] = $_POST['category_display'] ?? $config['category_display'];

    // Admin-Passwort ändern (optional)
    if (!empty($_POST['new_admin_password'])) {
        $config['admin_password'] = password_hash($_POST['new_admin_password'], PASSWORD_DEFAULT);
    }

    // Favicon Upload
    if (isset($_FILES['favicon']) && $_FILES['favicon']['error'] === UPLOAD_ERR_OK) {
        $file_ext = pathinfo($_FILES['favicon']['name'], PATHINFO_EXTENSION);
        $new_file_name = 'favicon.' . $file_ext; // Feste Benennung
        $target_file = $upload_dir . $new_file_name;
        if (move_uploaded_file($_FILES['favicon']['tmp_name'], $target_file)) {
            $config['favicon'] = $new_file_name;
        } else {
            $message .= '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Hochladen des Favicons.</div>';
        }
    }

    // Logo Upload
    if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
        $file_ext = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
        $new_file_name = 'logo.' . $file_ext; // Feste Benennung
        $target_file = $upload_dir . $new_file_name;
        if (move_uploaded_file($_FILES['logo']['tmp_name'], $target_file)) {
            $config['logo'] = $new_file_name;
        } else {
            $message .= '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Hochladen des Logos.</div>';
        }
    }

    // Handle image uploads for backgrounds
    $bg_types = ['page', 'category', 'linktree'];
    foreach ($bg_types as $type) {
        $file_input_name = "${type}_bg_value_file";
        if (isset($_POST["${type}_bg_type"]) && $_POST["${type}_bg_type"] === 'image' && isset($_FILES[$file_input_name]) && $_FILES[$file_input_name]['error'] === UPLOAD_ERR_OK) {
            $file_ext = pathinfo($_FILES[$file_input_name]['name'], PATHINFO_EXTENSION);
            $new_file_name = uniqid("${type}_bg_") . '.' . $file_ext;
            $target_file = $upload_dir . $new_file_name;
            if (move_uploaded_file($_FILES[$file_input_name]['tmp_name'], $target_file)) {
                $config['colors']["${type}_bg_value"] = $new_file_name;
            } else {
                $message .= '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Hochladen des Hintergrundbildes für ' . $type . '.</div>';
            }
        }
    }


    if (save_json($config_file, $config)) {
        $message .= '<div class="bg-green-500 text-white p-3 rounded-md mb-4">Allgemeine Einstellungen erfolgreich aktualisiert.</div>';
    } else {
        $message .= '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Speichern der allgemeinen Einstellungen.</div>';
    }
    $current_tab = 'general';
}

// --- Benutzerverwaltung (Users) ---
$users = load_json($users_file);
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_user'])) {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    if (!empty($username) && !empty($password)) {
        $users[] = ['id' => uniqid(), 'username' => $username, 'password' => password_hash($password, PASSWORD_DEFAULT)];
        if (save_json($users_file, $users)) {
            $message = '<div class="bg-green-500 text-white p-3 rounded-md mb-4">Benutzer erfolgreich hinzugefügt.</div>';
        } else {
            $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Hinzufügen des Benutzers.</div>';
        }
    } else {
        $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Benutzername und Passwort dürfen nicht leer sein.</div>';
    }
    $current_tab = 'users';
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_user'])) {
    $user_id = $_POST['user_id'] ?? '';
    $users = array_filter($users, fn($user) => $user['id'] !== $user_id);
    if (save_json($users_file, array_values($users))) {
        $message = '<div class="bg-green-500 text-white p-3 rounded-md mb-4">Benutzer erfolgreich gelöscht.</div>';
    } else {
        $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Löschen des Benutzers.</div>';
    }
    $current_tab = 'users';
}

// --- Kategorien (Categories) ---
$categories = load_json($categories_file);
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_category'])) {
    $category_name = $_POST['category_name'] ?? '';
    $category_icon = $_POST['category_icon'] ?? ''; // Neues Feld für Kategorie Icon
    if (!empty($category_name)) {
        $categories[] = ['id' => uniqid(), 'name' => $category_name, 'icon' => $category_icon]; // Icon speichern
        if (save_json($categories_file, $categories)) {
            $message = '<div class="bg-green-500 text-white p-3 rounded-md mb-4">Kategorie erfolgreich hinzugefügt.</div>';
        } else {
            $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Hinzufügen der Kategorie.</div>';
        }
    } else {
        $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Kategoriename darf nicht leer sein.</div>';
    }
    $current_tab = 'categories';
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_category'])) {
    $category_id = $_POST['category_id'] ?? '';
    $categories = array_filter($categories, fn($cat) => $cat['id'] !== $category_id);
    // Auch Links löschen, die zu dieser Kategorie gehören könnten (optional, hier nicht implementiert)
    if (save_json($categories_file, array_values($categories))) {
        $message = '<div class="bg-green-500 text-white p-3 rounded-md mb-4">Kategorie erfolgreich gelöscht.</div>';
    } else {
        $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Löschen der Kategorie.</div>';
    }
    $current_tab = 'categories';
}

// --- Links (Links) ---
$links = load_json($links_file);
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_link'])) {
    $link_data = [
        'id' => uniqid(),
        'name' => $_POST['link_name'] ?? '',
        'url' => $_POST['link_url'] ?? '',
        'category_id' => $_POST['link_category'] ?? '',
        'description' => $_POST['link_description'] ?? '',
        'password_protected' => isset($_POST['link_password_protected']) ? true : false,
        // Speichert Passwort im Klartext, wenn password_protected aktiviert ist
        'password' => isset($_POST['link_password_protected']) ? ($_POST['link_password'] ?? '') : '',
        'preview_enabled' => isset($_POST['link_preview_enabled']) ? true : false,
        'logo' => '',
    ];

    if (empty($link_data['name']) || empty($link_data['url'])) {
        $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Link-Name und URL dürfen nicht leer sein.</div>';
    } else {
        // Link-Logo Upload
        if (isset($_FILES['link_logo']) && $_FILES['link_logo']['error'] === UPLOAD_ERR_OK) {
            $file_ext = pathinfo($_FILES['link_logo']['name'], PATHINFO_EXTENSION);
            $new_file_name = uniqid('link_logo_') . '.' . $file_ext;
            $target_file = $upload_dir . $new_file_name;
            if (move_uploaded_file($_FILES['link_logo']['tmp_name'], $target_file)) {
                $link_data['logo'] = $new_file_name;
            } else {
                $message .= '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Hochladen des Link-Logos.</div>';
            }
        }

        $links[] = $link_data;
        if (save_json($links_file, $links)) {
            $message .= '<div class="bg-green-500 text-white p-3 rounded-md mb-4">Link erfolgreich hinzugefügt.</div>';
        } else {
            $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Hinzufügen des Links.</div>';
        }
    }
    $current_tab = 'links';
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_link'])) {
    $link_id = $_POST['link_id'] ?? '';
    $links = array_filter($links, fn($link) => $link['id'] !== $link_id);
    if (save_json($links_file, array_values($links))) {
        $message = '<div class="bg-green-500 text-white p-3 rounded-md mb-4">Link erfolgreich gelöscht.</div>';
    } else {
        $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Löschen des Links.</div>';
    }
    $current_tab = 'links';
}

// --- Texte (Texts) ---
$texts = load_json($texts_file);
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_text'])) {
    $text_content = $_POST['text_content'] ?? '';
    $text_category_id = $_POST['text_category'] ?? '';
    if (!empty($text_content)) {
        $texts[] = ['id' => uniqid(), 'content' => $text_content, 'category_id' => $text_category_id];
        if (save_json($texts_file, $texts)) {
            $message = '<div class="bg-green-500 text-white p-3 rounded-md mb-4">Text erfolgreich hinzugefügt.</div>';
        } else {
            $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Hinzufügen des Textes.</div>';
        }
    } else {
        $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Textinhalt darf nicht leer sein.</div>';
    }
    $current_tab = 'texts';
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_text'])) {
    $text_id = $_POST['text_id'] ?? '';
    $texts = array_filter($texts, fn($text) => $text['id'] !== $text_id);
    if (save_json($texts_file, array_values($texts))) {
        $message = '<div class="bg-green-500 text-white p-3 rounded-md mb-4">Text erfolgreich gelöscht.</div>';
    } else {
        $message = '<div class="bg-red-500 text-white p-3 rounded-md mb-4">Fehler beim Löschen des Textes.</div>';
    }
    $current_tab = 'texts';
}

// Hilfsfunktion, um den Pfad zu einem Upload zu erhalten
function get_upload_path($filename) {
    global $upload_dir;
    return file_exists($upload_dir . $filename) && !empty($filename) ? $upload_dir . $filename : '';
}

?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin-Panel - DigiLinks</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f0f2f5;
            min-height: 100vh;
        }
        .admin-container {
            max-width: 1200px;
            margin: 2rem auto;
            background-color: #ffffff;
            border-radius: 0.75rem;
            box-shadow: 0 10px 15px rgba(0, 0, 0, 0.1);
            padding: 2rem;
        }
        .tab-button {
            padding: 0.75rem 1.5rem;
            border-radius: 0.5rem 0.5rem 0 0;
            font-weight: 600;
            color: #4a5568;
            background-color: #e2e8f0;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        .tab-button.active {
            background-color: #ffffff;
            color: #4c51bf;
            border-bottom: 2px solid #4c51bf;
        }
        .tab-content {
            padding: 1.5rem;
            border-top: 1px solid #e2e8f0;
        }
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: #4a5568;
        }
        .form-group input[type="text"],
        .form-group input[type="password"],
        .form-group input[type="url"],
        .form-group input[type="color"],
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #cbd5e0;
            border-radius: 0.375rem;
            margin-bottom: 1rem;
            box-sizing: border-box;
        }
        .form-group input[type="file"] {
            margin-bottom: 1rem;
        }
        .form-group button {
            padding: 0.75rem 1.5rem;
            background-color: #4c51bf;
            color: white;
            border: none;
            border-radius: 0.375rem;
            font-size: 1rem;
            cursor: pointer;
            transition: background-color 0.3s ease;
        }
        .form-group button:hover {
            background-color: #434190;
        }
        .delete-button {
            background-color: #ef4444;
        }
        .delete-button:hover {
            background-color: #dc2626;
        }
        .table-container {
            overflow-x: auto;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1rem;
        }
        th, td {
            border: 1px solid #e2e8f0;
            padding: 0.75rem;
            text-align: left;
        }
        th {
            background-color: #f7fafc;
            font-weight: 600;
            color: #2d3748;
        }
        .gradient-preview {
            width: 100%;
            height: 40px;
            border: 1px solid #cbd5e0;
            border-radius: 0.375rem;
            margin-top: 0.5rem;
            margin-bottom: 1rem;
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <div class="flex justify-between items-center mb-6">
            <h1 class="text-3xl font-bold text-gray-800">Admin-Panel</h1>
            <a href="?action=logout" class="text-red-500 hover:text-red-700 font-semibold">Abmelden</a>
        </div>

        <?php echo $message; // Zeigt Erfolgs- oder Fehlermeldungen an ?>

        <div class="tabs mb-6">
            <div class="flex border-b border-gray-200">
                <button class="tab-button <?php echo ($current_tab === 'general' ? 'active' : ''); ?>" onclick="showTab('general')">Allgemein</button>
                <button class="tab-button <?php echo ($current_tab === 'users' ? 'active' : ''); ?>" onclick="showTab('users')">Benutzer</button>
                <button class="tab-button <?php echo ($current_tab === 'categories' ? 'active' : ''); ?>" onclick="showTab('categories')">Kategorien</button>
                <button class="tab-button <?php echo ($current_tab === 'links' ? 'active' : ''); ?>" onclick="showTab('links')">Links</button>
                <button class="tab-button <?php echo ($current_tab === 'texts' ? 'active' : ''); ?>" onclick="showTab('texts')">Texte</button>
            </div>

            <!-- Tab: Allgemeine Einstellungen -->
            <div id="tab-general" class="tab-content <?php echo ($current_tab === 'general' ? '' : 'hidden'); ?>">
                <h2 class="text-2xl font-semibold text-gray-700 mb-4">Allgemeine Einstellungen</h2>
                <form method="POST" enctype="multipart/form-data">
                    <div class="form-group">
                        <label for="linktree_name">Linktree Name:</label>
                        <input type="text" id="linktree_name" name="linktree_name" value="<?php echo htmlspecialchars($config['linktree_name'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label for="linktree_icon">Linktree Icon (Font Awesome Class, optional):</label>
                        <input type="text" id="linktree_icon" name="linktree_icon" value="<?php echo htmlspecialchars($config['linktree_icon'] ?? ''); ?>" placeholder="z.B. fa-solid fa-tree">
                        <p class="text-sm text-gray-500 mt-1">Geben Sie eine Font Awesome Klasse ein (z.B. fa-solid fa-house). Icons finden Sie hier: <a href="https://fontawesome.com/icons?d=gallery&m=free" target="_blank" class="text-blue-500 hover:underline">Font Awesome Free Icons</a></p>
                    </div>
                    <div class="form-group">
                        <label for="new_admin_password">Neues Admin Passwort (optional):</label>
                        <input type="password" id="new_admin_password" name="new_admin_password" placeholder="Lassen Sie es leer, um es nicht zu ändern">
                    </div>
                    <div class="form-group">
                        <label for="favicon">Favicon (.ico, .png, .svg):</label>
                        <input type="file" id="favicon" name="favicon" accept=".ico,.png,.svg">
                        <?php if (!empty($config['favicon']) && file_exists($upload_dir . $config['favicon'])): ?>
                            <img src="<?php echo $upload_dir . htmlspecialchars($config['favicon']); ?>" alt="Favicon" class="w-10 h-10 rounded-md mt-2">
                            <span class="text-sm text-gray-500">Aktuelles Favicon</span>
                        <?php endif; ?>
                    </div>
                    <div class="form-group">
                        <label for="logo">Logo (.png, .jpg, .jpeg, .svg):</label>
                        <input type="file" id="logo" name="logo" accept=".png,.jpg,.jpeg,.svg">
                        <?php if (!empty($config['logo']) && file_exists($upload_dir . $config['logo'])): ?>
                            <img src="<?php echo $upload_dir . htmlspecialchars($config['logo']); ?>" alt="Logo" class="w-20 h-20 rounded-md mt-2">
                            <span class="text-sm text-gray-500">Aktuelles Logo</span>
                        <?php endif; ?>
                    </div>

                    <h3 class="text-xl font-semibold text-gray-700 mb-3 mt-6">Farben anpassen</h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div class="form-group">
                            <label for="page_bg_type">Hintergrund Seite Typ:</label>
                            <select id="page_bg_type" name="page_bg_type" onchange="toggleBgValue('page')">
                                <option value="solid" <?php echo (($config['colors']['page_bg_type'] ?? '') === 'solid' ? 'selected' : ''); ?>>Einfarbig</option>
                                <option value="gradient" <?php echo (($config['colors']['page_bg_type'] ?? '') === 'gradient' ? 'selected' : ''); ?>>Verlauf</option>
                                <option value="image" <?php echo (($config['colors']['page_bg_type'] ?? '') === 'image' ? 'selected' : ''); ?>>Bild</option>
                            </select>
                        </div>
                        <div class="form-group" id="page_bg_value_container">
                            <label for="page_bg_value">Hintergrund Seite Wert:</label>
                            <?php if (($config['colors']['page_bg_type'] ?? '') === 'image'): ?>
                                <input type="file" id="page_bg_value_file" name="page_bg_value_file" accept="image/*">
                                <?php if (!empty($config['colors']['page_bg_value']) && file_exists($upload_dir . $config['colors']['page_bg_value'])): ?>
                                    <span class="text-sm text-gray-500">Aktuelles Bild: <?php echo htmlspecialchars($config['colors']['page_bg_value']); ?></span>
                                <?php endif; ?>
                            <?php elseif (($config['colors']['page_bg_type'] ?? '') === 'solid'): ?>
                                <input type="color" id="page_bg_value" name="page_bg_value" value="<?php echo htmlspecialchars($config['colors']['page_bg_value'] ?? '#1a202c'); ?>">
                            <?php else: // gradient ?>
                                <input type="text" id="page_bg_value" name="page_bg_value" value="<?php echo htmlspecialchars($config['colors']['page_bg_value'] ?? '#1a202c'); ?>" placeholder="CSS-Verlauf (z.B. linear-gradient(to right, #ff0000, #0000ff))" oninput="updateGradientPreview('page')">
                                <div id="page_bg_preview" class="gradient-preview" style="<?php echo (($config['colors']['page_bg_type'] ?? '') === 'gradient' ? 'background: ' . htmlspecialchars($config['colors']['page_bg_value'] ?? '') : ''); ?>"></div>
                                <p class="text-sm text-gray-500 mt-2">Der CSS-Verlauf kann über Generatoren erstellt werden. Ein Beispiel ist: <a href="https://cssgradient.io" target="_blank" class="text-blue-500 hover:underline">https://cssgradient.io</a></p>
                            <?php endif; ?>
                        </div>

                        <div class="form-group">
                            <label for="category_bg_type">Hintergrund Kategorie Typ:</label>
                            <select id="category_bg_type" name="category_bg_type" onchange="toggleBgValue('category')">
                                <option value="solid" <?php echo (($config['colors']['category_bg_type'] ?? '') === 'solid' ? 'selected' : ''); ?>>Einfarbig</option>
                                <option value="gradient" <?php echo (($config['colors']['category_bg_type'] ?? '') === 'gradient' ? 'selected' : ''); ?>>Verlauf</option>
                                <option value="image" <?php echo (($config['colors']['category_bg_type'] ?? '') === 'image' ? 'selected' : ''); ?>>Bild</option>
                            </select>
                        </div>
                        <div class="form-group" id="category_bg_value_container">
                            <label for="category_bg_value">Hintergrund Kategorie Wert:</label>
                            <?php if (($config['colors']['category_bg_type'] ?? '') === 'image'): ?>
                                <input type="file" id="category_bg_value_file" name="category_bg_value_file" accept="image/*">
                                <?php if (!empty($config['colors']['category_bg_value']) && file_exists($upload_dir . $config['colors']['category_bg_value'])): ?>
                                    <span class="text-sm text-gray-500">Aktuelles Bild: <?php echo htmlspecialchars($config['colors']['category_bg_value']); ?></span>
                                <?php endif; ?>
                            <?php elseif (($config['colors']['category_bg_type'] ?? '') === 'solid'): ?>
                                <input type="color" id="category_bg_value" name="category_bg_value" value="<?php echo htmlspecialchars($config['colors']['category_bg_value'] ?? '#2d3748'); ?>">
                            <?php else: // gradient ?>
                                <input type="text" id="category_bg_value" name="category_bg_value" value="<?php echo htmlspecialchars($config['colors']['category_bg_value'] ?? '#2d3748'); ?>" placeholder="CSS-Verlauf (z.B. linear-gradient(to right, #ff0000, #0000ff))" oninput="updateGradientPreview('category')">
                                <div id="category_bg_preview" class="gradient-preview" style="<?php echo (($config['colors']['category_bg_type'] ?? '') === 'gradient' ? 'background: ' . htmlspecialchars($config['colors']['category_bg_value'] ?? '') : ''); ?>"></div>
                                <p class="text-sm text-gray-500 mt-2">Der CSS-Verlauf kann über Generatoren erstellt werden. Ein Beispiel ist: <a href="https://cssgradient.io" target="_blank" class="text-blue-500 hover:underline">https://cssgradient.io</a></p>
                            <?php endif; ?>
                        </div>

                        <div class="form-group">
                            <label for="linktree_bg_type">Hintergrund Linktree Gesamt Typ:</label>
                            <select id="linktree_bg_type" name="linktree_bg_type" onchange="toggleBgValue('linktree')">
                                <option value="solid" <?php echo (($config['colors']['linktree_bg_type'] ?? '') === 'solid' ? 'selected' : ''); ?>>Einfarbig</option>
                                <option value="gradient" <?php echo (($config['colors']['linktree_bg_type'] ?? '') === 'gradient' ? 'selected' : ''); ?>>Verlauf</option>
                                <option value="image" <?php echo (($config['colors']['linktree_bg_type'] ?? '') === 'image' ? 'selected' : ''); ?>>Bild</option>
                            </select>
                        </div>
                        <div class="form-group" id="linktree_bg_value_container">
                            <label for="linktree_bg_value">Hintergrund Linktree Gesamt Wert:</label>
                            <?php if (($config['colors']['linktree_bg_type'] ?? '') === 'image'): ?>
                                <input type="file" id="linktree_bg_value_file" name="linktree_bg_value_file" accept="image/*">
                                <?php if (!empty($config['colors']['linktree_bg_value']) && file_exists($upload_dir . $config['colors']['linktree_bg_value'])): ?>
                                    <span class="text-sm text-gray-500">Aktuelles Bild: <?php echo htmlspecialchars($config['colors']['linktree_bg_value']); ?></span>
                                <?php endif; ?>
                            <?php elseif (($config['colors']['linktree_bg_type'] ?? '') === 'solid'): ?>
                                <input type="color" id="linktree_bg_value" name="linktree_bg_value" value="<?php echo htmlspecialchars($config['colors']['linktree_bg_value'] ?? '#4a5568'); ?>">
                            <?php else: // gradient ?>
                                <input type="text" id="linktree_bg_value" name="linktree_bg_value" value="<?php echo htmlspecialchars($config['colors']['linktree_bg_value'] ?? '#4a5568'); ?>" placeholder="CSS-Verlauf (z.B. linear-gradient(to right, #ff0000, #0000ff))" oninput="updateGradientPreview('linktree')">
                                <div id="linktree_bg_preview" class="gradient-preview" style="<?php echo (($config['colors']['linktree_bg_type'] ?? '') === 'gradient' ? 'background: ' . htmlspecialchars($config['colors']['linktree_bg_value'] ?? '') : ''); ?>"></div>
                                <p class="text-sm text-gray-500 mt-2">Der CSS-Verlauf kann über Generatoren erstellt werden. Ein Beispiel ist: <a href="https://cssgradient.io" target="_blank" class="text-blue-500 hover:underline">https://cssgradient.io</a></p>
                            <?php endif; ?>
                        </div>
                        <div class="form-group">
                            <label for="text_color">Schriftfarbe:</label>
                            <input type="color" id="text_color" name="text_color" value="<?php echo htmlspecialchars($config['colors']['text'] ?? '#ffffff'); ?>">
                        </div>
                    </div>

                    <h3 class="text-xl font-semibold text-gray-700 mb-3 mt-6">Kategorien-Anzeige</h3>
                    <div class="form-group">
                        <label for="category_display">Kategorien anzeigen als:</label>
                        <select id="category_display" name="category_display">
                            <option value="stacked" <?php echo (($config['category_display'] ?? '') === 'stacked' ? 'selected' : ''); ?>>Untereinander (Gestapelt)</option>
                            <option value="tabs" <?php echo (($config['category_display'] ?? '') === 'tabs' ? 'selected' : ''); ?>>Registerkarten</option>
                            <option value="dropdown" <?php echo (($config['category_display'] ?? '') === 'dropdown' ? 'selected' : ''); ?>>Dropdown</option>
                        </select>
                    </div>

                    <div class="form-group mt-6">
                        <button type="submit" name="update_general_settings">Einstellungen speichern</button>
                    </div>
                </form>
            </div>

            <!-- Tab: Benutzer -->
            <div id="tab-users" class="tab-content <?php echo ($current_tab === 'users' ? '' : 'hidden'); ?>">
                <h2 class="text-2xl font-semibold text-gray-700 mb-4">Benutzerverwaltung</h2>
                <form method="POST" class="mb-6">
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4 items-end">
                        <div class="form-group">
                            <label for="username">Benutzername:</label>
                            <input type="text" id="username" name="username" required>
                        </div>
                        <div class="form-group">
                            <label for="password">Passwort:</label>
                            <input type="password" id="password" name="password" required>
                        </div>
                        <div class="form-group">
                            <button type="submit" name="add_user">Benutzer hinzufügen</button>
                        </div>
                    </div>
                </form>

                <h3 class="text-xl font-semibold text-gray-700 mb-3">Vorhandene Benutzer</h3>
                <div class="table-container">
                    <table class="min-w-full bg-white rounded-md">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Benutzername</th>
                                <th>Aktionen</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($users as $user): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($user['id']); ?></td>
                                    <td><?php echo htmlspecialchars($user['username']); ?></td>
                                    <td>
                                        <form method="POST" class="inline-block">
                                            <input type="hidden" name="user_id" value="<?php echo htmlspecialchars($user['id']); ?>">
                                            <button type="submit" name="delete_user" class="delete-button px-3 py-1 text-sm rounded-md">Löschen</button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            <?php if (empty($users)): ?>
                                <tr><td colspan="3" class="text-center text-gray-500">Keine Benutzer vorhanden.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Tab: Kategorien -->
            <div id="tab-categories" class="tab-content <?php echo ($current_tab === 'categories' ? '' : 'hidden'); ?>">
                <h2 class="text-2xl font-semibold text-gray-700 mb-4">Kategorienverwaltung</h2>
                <form method="POST" class="mb-6">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 items-end">
                        <div class="form-group">
                            <label for="category_name">Kategoriename:</label>
                            <input type="text" id="category_name" name="category_name" required>
                        </div>
                        <div class="form-group">
                            <label for="category_icon">Kategorie Icon (Font Awesome Class, optional):</label>
                            <input type="text" id="category_icon" name="category_icon" placeholder="z.B. fa-solid fa-folder">
                            <p class="text-sm text-gray-500 mt-1">Geben Sie eine Font Awesome Klasse ein (z.B. fa-solid fa-folder). Icons finden Sie hier: <a href="https://fontawesome.com/icons?d=gallery&m=free" target="_blank" class="text-blue-500 hover:underline">Font Awesome Free Icons</a></p>
                        </div>
                        <div class="form-group">
                            <button type="submit" name="add_category">Kategorie hinzufügen</button>
                        </div>
                    </div>
                </form>

                <h3 class="text-xl font-semibold text-gray-700 mb-3">Vorhandene Kategorien</h3>
                <div class="table-container">
                    <table class="min-w-full bg-white rounded-md">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Name</th>
                                <th>Icon</th>
                                <th>Aktionen</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($categories as $category): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($category['id']); ?></td>
                                    <td><?php echo htmlspecialchars($category['name']); ?></td>
                                    <td>
                                        <?php if (!empty($category['icon'])): ?>
                                            <i class="<?php echo htmlspecialchars($category['icon']); ?>"></i>
                                        <?php else: ?>
                                            -
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <form method="POST" class="inline-block">
                                            <input type="hidden" name="category_id" value="<?php echo htmlspecialchars($category['id']); ?>">
                                            <button type="submit" name="delete_category" class="delete-button px-3 py-1 text-sm rounded-md">Löschen</button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            <?php if (empty($categories)): ?>
                                <tr><td colspan="4" class="text-center text-gray-500">Keine Kategorien vorhanden.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Tab: Links -->
            <div id="tab-links" class="tab-content <?php echo ($current_tab === 'links' ? '' : 'hidden'); ?>">
                <h2 class="text-2xl font-semibold text-gray-700 mb-4">Links verwalten</h2>
                <form method="POST" enctype="multipart/form-data" class="mb-6">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div class="form-group">
                            <label for="link_name">Link Name:</label>
                            <input type="text" id="link_name" name="link_name" required>
                        </div>
                        <div class="form-group">
                            <label for="link_url">Link URL:</label>
                            <input type="url" id="link_url" name="link_url" required>
                        </div>
                        <div class="form-group">
                            <label for="link_category">Kategorie:</label>
                            <select id="link_category" name="link_category">
                                <option value="">Keine Kategorie</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo htmlspecialchars($category['id']); ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="link_description">Beschreibung (optional):</label>
                            <textarea id="link_description" name="link_description" rows="3" class="w-full p-2 border rounded"></textarea>
                        </div>
                        <div class="form-group">
                            <label for="link_logo">Link Logo (optional, .png, .jpg, .jpeg, .svg):</label>
                            <input type="file" id="link_logo" name="link_logo" accept=".png,.jpg,.jpeg,.svg">
                        </div>
                        <div class="form-group flex items-center">
                            <input type="checkbox" id="link_password_protected" name="link_password_protected" class="mr-2" onchange="togglePasswordInput()">
                            <label for="link_password_protected" class="mb-0">Passwortgeschützt?</label>
                        </div>
                        <div class="form-group" id="link_password_container" style="display: none;">
                            <label for="link_password">Link Passwort:</label>
                            <input type="password" id="link_password" name="link_password">
                        </div>
                        <div class="form-group flex items-center">
                            <input type="checkbox" id="link_preview_enabled" name="link_preview_enabled" class="mr-2">
                            <label for="link_preview_enabled" class="mb-0">Vorschau aktivieren?</label>
                        </div>
                    </div>
                    <div class="form-group mt-6">
                        <button type="submit" name="add_link">Link hinzufügen</button>
                    </div>
                </form>

                <h3 class="text-xl font-semibold text-gray-700 mb-3">Vorhandene Links</h3>
                <div class="table-container">
                    <table class="min-w-full bg-white rounded-md">
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>URL</th>
                                <th>Kategorie</th>
                                <th>Passwort</th>
                                <th>Vorschau</th>
                                <th>Logo</th>
                                <th>Aktionen</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($links as $link): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($link['name']); ?></td>
                                    <td><a href="<?php echo htmlspecialchars($link['url']); ?>" target="_blank" class="text-blue-500 hover:underline"><?php echo htmlspecialchars($link['url']); ?></a></td>
                                    <td>
                                        <?php
                                        $category_name = 'Keine';
                                        foreach ($categories as $cat) {
                                            if ($cat['id'] === ($link['category_id'] ?? '')) {
                                                $category_name = $cat['name'];
                                                break;
                                            }
                                        }
                                        echo htmlspecialchars($category_name);
                                        ?>
                                    </td>
                                    <td><?php echo ($link['password_protected'] ? 'Ja' : 'Nein'); ?></td>
                                    <td><?php echo ($link['preview_enabled'] ? 'Ja' : 'Nein'); ?></td>
                                    <td>
                                        <?php if (!empty($link['logo']) && file_exists($upload_dir . $link['logo'])): ?>
                                            <img src="<?php echo $upload_dir . htmlspecialchars($link['logo']); ?>" alt="Link Logo" class="w-8 h-8 rounded-full">
                                        <?php else: ?>
                                            Kein Logo
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <form method="POST" class="inline-block">
                                            <input type="hidden" name="link_id" value="<?php echo htmlspecialchars($link['id']); ?>">
                                            <button type="submit" name="delete_link" class="delete-button px-3 py-1 text-sm rounded-md">Löschen</button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            <?php if (empty($links)): ?>
                                <tr><td colspan="7" class="text-center text-gray-500">Keine Links vorhanden.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Tab: Texte -->
            <div id="tab-texts" class="tab-content <?php echo ($current_tab === 'texts' ? '' : 'hidden'); ?>">
                <h2 class="text-2xl font-semibold text-gray-700 mb-4">Texte verwalten</h2>
                <form method="POST" class="mb-6">
                    <div class="form-group">
                        <label for="text_content">Textinhalt:</label>
                        <textarea id="text_content" name="text_content" rows="5" class="w-full p-2 border rounded" required></textarea>
                    </div>
                    <div class="form-group">
                        <label for="text_category">Kategorie (optional):</label>
                        <select id="text_category" name="text_category">
                            <option value="">Keine Kategorie</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo htmlspecialchars($category['id']); ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group mt-6">
                        <button type="submit" name="add_text">Text hinzufügen</button>
                    </div>
                </form>

                <h3 class="text-xl font-semibold text-gray-700 mb-3">Vorhandene Texte</h3>
                <div class="table-container">
                    <table class="min-w-full bg-white rounded-md">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Inhalt</th>
                                <th>Kategorie</th>
                                <th>Aktionen</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($texts as $text): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($text['id']); ?></td>
                                    <td><?php echo htmlspecialchars(substr($text['content'], 0, 100)) . (strlen($text['content']) > 100 ? '...' : ''); ?></td>
                                    <td>
                                        <?php
                                        $category_name = 'Keine';
                                        foreach ($categories as $cat) {
                                            if ($cat['id'] === ($text['category_id'] ?? '')) {
                                                $category_name = $cat['name'];
                                                break;
                                            }
                                        }
                                        echo htmlspecialchars($category_name);
                                        ?>
                                    </td>
                                    <td>
                                        <form method="POST" class="inline-block">
                                            <input type="hidden" name="text_id" value="<?php echo htmlspecialchars($text['id']); ?>">
                                            <button type="submit" name="delete_text" class="delete-button px-3 py-1 text-sm rounded-md">Löschen</button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            <?php if (empty($texts)): ?>
                                <tr><td colspan="4" class="text-center text-gray-500">Keine Texte vorhanden.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Pass the initial configuration colors and upload directory from PHP to JavaScript
        const initialConfigColors = <?php echo json_encode($config['colors'] ?? []); ?>;
        const uploadDir = "<?php echo $upload_dir; ?>"; // Pass upload directory for image paths

        // Funktion zum Wechseln der Tabs
        function showTab(tabId) {
            const tabs = document.querySelectorAll('.tab-content');
            tabs.forEach(tab => tab.classList.add('hidden'));
            document.getElementById('tab-' + tabId).classList.remove('hidden');

            const buttons = document.querySelectorAll('.tab-button');
            buttons.forEach(button => button.classList.remove('active'));
            document.querySelector(`.tab-button[onclick="showTab('${tabId}')"]`).classList.add('active');

            // URL aktualisieren, um den aktuellen Tab zu speichern
            history.pushState(null, '', `admin.php?tab=${tabId}`);
        }

        // Funktion zum Umschalten des Passwort-Eingabefeldes für Links
        function togglePasswordInput() {
            const checkbox = document.getElementById('link_password_protected');
            const passwordContainer = document.getElementById('link_password_container');
            if (checkbox.checked) {
                passwordContainer.style.display = 'block';
            } else {
                passwordContainer.style.display = 'none';
            }
        }

        // Funktion zum Aktualisieren der Gradienten-Vorschau
        function updateGradientPreview(type) {
            const inputElement = document.getElementById(`${type}_bg_value`);
            const previewElement = document.getElementById(`${type}_bg_preview`);
            if (inputElement && previewElement) {
                try {
                    previewElement.style.background = inputElement.value;
                } catch (e) {
                    // Ungültiger CSS-Verlauf, Vorschau zurücksetzen oder Fehlermeldung anzeigen
                    previewElement.style.background = 'none';
                }
            }
        }

        // Funktion zum Umschalten des Hintergrund-Wert-Eingabefeldes basierend auf dem Typ
        function toggleBgValue(type) {
            const select = document.getElementById(`${type}_bg_type`);
            const container = document.getElementById(`${type}_bg_value_container`);
            
            // Versuche, den aktuellen Wert aus dem Eingabefeld zu erhalten, falls vorhanden
            let currentValue = '';
            const existingInput = container.querySelector('input[type="text"], input[type="color"]');
            if (existingInput) {
                currentValue = existingInput.value;
            } else {
                // Fallback: Wenn kein Eingabefeld existiert (z.g. beim Wechsel von Bild zu Farbe/Verlauf),
                // versuche, den Wert aus der initialen Konfiguration zu holen.
                currentValue = initialConfigColors[`${type}_bg_value`] || '';
            }

            let inputHtml = '';
            let previewHtml = '';
            let infoTextHtml = '';

            if (select.value === 'image') {
                inputHtml = `
                    <input type="file" id="${type}_bg_value_file" name="${type}_bg_value_file" accept="image/*">
                `;
                // Zeige den aktuellen Bildpfad an, falls er in der initialen Konfiguration als Bild gesetzt war
                if (initialConfigColors[`${type}_bg_type`] === 'image' && initialConfigColors[`${type}_bg_value`]) {
                    inputHtml += `<span class="text-sm text-gray-500">Aktuelles Bild: ${initialConfigColors[`${type}_bg_value`]}</span>`;
                }
            } else if (select.value === 'solid') {
                // Standardwert für Farbregler ist Schwarz, falls kein Wert vorhanden
                inputHtml = `<input type="color" id="${type}_bg_value" name="${type}_bg_value" value="${currentValue || '#000000'}">`;
            } else { // gradient
                // Standardwert für Verlauf, falls kein Wert vorhanden
                inputHtml = `
                    <input type="text" id="${type}_bg_value" name="${type}_bg_value" value="${currentValue || 'linear-gradient(to right, #ff0000, #0000ff)'}" placeholder="CSS-Verlauf (z.B. linear-gradient(to right, #ff0000, #0000ff))" oninput="updateGradientPreview('${type}')">
                `;
                // Vorschau-Div für Verlauf
                previewHtml = `<div id="${type}_bg_preview" class="gradient-preview" style="background: ${currentValue || 'none'}"></div>`;
                // Info-Text für Verlauf
                infoTextHtml = `<p class="text-sm text-gray-500 mt-2">Der CSS-Verlauf kann über Generatoren erstellt werden. Ein Beispiel ist: <a href="https://cssgradient.io" target="_blank" class="text-blue-500 hover:underline">https://cssgradient.io</a></p>`;
            }
            container.innerHTML = `<label for="${type}_bg_value">Hintergrund ${type.charAt(0).toUpperCase() + type.slice(1)} Wert:</label>${inputHtml}${previewHtml}${infoTextHtml}`;

            // Wenn der Typ "gradient" ist, die Vorschau beim Umschalten sofort aktualisieren
            if (select.value === 'gradient') {
                updateGradientPreview(type);
            }
        }


        // Initialisierung beim Laden der Seite
        document.addEventListener('DOMContentLoaded', () => {
            // Aktiven Tab beim Laden der Seite anzeigen
            const urlParams = new URLSearchParams(window.location.search);
            const initialTab = urlParams.get('tab') || 'general';
            showTab(initialTab);

            // Initialen Zustand für Passwort-Feld setzen
            togglePasswordInput();

            // Initialen Zustand für Hintergrund-Wert-Felder setzen
            // Manuell für jeden Typ aufrufen, um den korrekten Input-Typ beim Laden zu gewährleisten
            // Dies ist wichtig, da die PHP-Logik nur den initialen Zustand rendert.
            // Die JS-Funktion muss dann die Elemente basierend auf dem Typ korrekt initialisieren.
            toggleBgValue('page');
            toggleBgValue('category');
            toggleBgValue('linktree');
        });
    </script>
</body>
</html>
