<?php
// Pfade zu den JSON-Dateien
$config_file = 'config.json';
$categories_file = 'categories.json';
$links_file = 'links.json';
$texts_file = 'texts.json';
$upload_dir = 'uploads/';

// Funktionen zum Laden von JSON-Daten
function load_json($file) {
    if (file_exists($file)) {
        return json_decode(file_get_contents($file), true);
    }
    return [];
}

// Konfiguration und Daten laden
$config = load_json($config_file);
$categories = load_json($categories_file);
$links = load_json($links_file);
$texts = load_json($texts_file);

// Standardwerte, falls Konfiguration nicht existiert (z.B. wenn Setup noch nicht gelaufen ist)
$linktree_name = $config['linktree_name'] ?? 'Mein Linktree';
$linktree_icon = $config['linktree_icon'] ?? '';
$favicon_path = !empty($config['favicon']) && file_exists($upload_dir . $config['favicon']) ? $upload_dir . htmlspecialchars($config['favicon']) : '';
$logo_path = !empty($config['logo']) && file_exists($upload_dir . $config['logo']) ? $upload_dir . htmlspecialchars($config['logo']) : '';

$text_color = $config['colors']['text'] ?? '#ffffff';

$page_bg_type = $config['colors']['page_bg_type'] ?? 'solid';
$page_bg_value = $config['colors']['page_bg_value'] ?? '#1a202c';
if ($page_bg_type === 'image' && !empty($page_bg_value) && file_exists($upload_dir . $page_bg_value)) {
    $page_bg_value = 'url(\'' . $upload_dir . htmlspecialchars($page_bg_value) . '\')';
}

$category_bg_type = $config['colors']['category_bg_type'] ?? 'solid';
$category_bg_value = $config['colors']['category_bg_value'] ?? '#2d3748';
if ($category_bg_type === 'image' && !empty($category_bg_value) && file_exists($upload_dir . $category_bg_value)) {
    $category_bg_value = 'url(\'' . $upload_dir . htmlspecialchars($category_bg_value) . '\')';
}

$linktree_bg_type = $config['colors']['linktree_bg_type'] ?? 'solid';
$linktree_bg_value = $config['colors']['linktree_bg_value'] ?? '#4a5568';
if ($linktree_bg_type === 'image' && !empty($linktree_bg_value) && file_exists($upload_dir . $linktree_bg_value)) {
    $linktree_bg_value = 'url(\'' . $upload_dir . htmlspecialchars($linktree_bg_value) . '\')';
}


$category_display = $config['category_display'] ?? 'stacked'; // stacked, tabs, or dropdown

// Links und Texte nach Kategorien gruppieren
$categorized_content = [];

// Füge eine 'uncategorized' Kategorie hinzu, falls es Inhalte ohne Kategorie gibt
$has_uncategorized_content = false;
foreach ($links as $link) {
    if (!isset($link['category_id']) || empty($link['category_id'])) {
        $has_uncategorized_content = true;
        break;
    }
}
if (!$has_uncategorized_content) {
    foreach ($texts as $text) {
        if (!isset($text['category_id']) || empty($text['category_id'])) {
            $has_uncategorized_content = true;
            break;
        }
    }
}

if ($has_uncategorized_content) {
    $categorized_content['uncategorized'] = [
        'name' => 'Unkategorisiert',
        'icon' => 'fa-solid fa-question-circle',
        'links' => [],
        'texts' => []
    ];
}

// Initialisiere Kategorien aus der JSON-Datei
foreach ($categories as $category) {
    $categorized_content[$category['id']] = [
        'name' => $category['name'],
        'icon' => $category['icon'] ?? '',
        'links' => [],
        'texts' => []
    ];
}

// Links zuordnen
foreach ($links as $link) {
    $category_id = $link['category_id'] ?? '';
    if (isset($categorized_content[$category_id])) {
        $categorized_content[$category_id]['links'][] = $link;
    } else {
        // Wenn category_id ungültig oder fehlt, der 'uncategorized' Kategorie zuordnen
        if (isset($categorized_content['uncategorized'])) {
            $categorized_content['uncategorized']['links'][] = $link;
        }
    }
}

// Texte zuordnen
foreach ($texts as $text) {
    $category_id = $text['category_id'] ?? '';
    if (isset($categorized_content[$category_id])) {
        $categorized_content[$category_id]['texts'][] = $text;
    } else {
        if (isset($categorized_content['uncategorized'])) {
            $categorized_content['uncategorized']['texts'][] = $text;
        }
    }
}

// Leere Kategorien entfernen (außer 'uncategorized', wenn es Inhalte hat)
foreach ($categorized_content as $id => $content) {
    if ($id !== 'uncategorized' && empty($content['links']) && empty($content['texts'])) {
        unset($categorized_content[$id]);
    }
}
// 'uncategorized' entfernen, wenn es nach der Zuordnung leer ist
if (isset($categorized_content['uncategorized']) && empty($categorized_content['uncategorized']['links']) && empty($categorized_content['uncategorized']['texts'])) {
    unset($categorized_content['uncategorized']);
}

// Füge 'all' Kategorie hinzu (unabhängig vom Display-Modus)
$all_links_combined = [];
$all_texts_combined = [];
foreach ($categorized_content as $cat_id => $content) {
    if ($cat_id !== 'all') { // Vermeide Endlosschleife, falls 'all' irgendwie verarbeitet wird
        $all_links_combined = array_merge($all_links_combined, $content['links']);
        $all_texts_combined = array_merge($all_texts_combined, $content['texts']);
    }
}

// Füge 'all' nur hinzu, wenn es tatsächlich Inhalte über Kategorien hinweg gibt
if (!empty($all_links_combined) || !empty($all_texts_combined)) {
    $categorized_content['all'] = [
        'name' => 'Alle Kategorien',
        'icon' => '',
        'links' => $all_links_combined,
        'texts' => $all_texts_combined
    ];
}

// Kategorien sortieren, um 'all' zuerst zu platzieren
uksort($categorized_content, function($a, $b) {
    if ($a === 'all') return -1;
    if ($b === 'all') return 1;
    return 0;
});

?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($linktree_name); ?></title>
    <?php if (!empty($favicon_path)): ?>
        <link rel="icon" href="<?php echo $favicon_path; ?>" type="image/x-icon">
    <?php endif; ?>
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Font Awesome CDN für Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            color: <?php echo $text_color; ?>;
            <?php if ($page_bg_type === 'solid'): ?>
                background-color: <?php echo $page_bg_value; ?>;
            <?php elseif ($page_bg_type === 'gradient'): ?>
                background: <?php echo $page_bg_value; ?>;
            <?php elseif ($page_bg_type === 'image'): ?>
                background-image: <?php echo $page_bg_value; ?>;
                background-size: cover;
                background-position: center;
                background-attachment: fixed;
            <?php endif; ?>
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 1rem;
        }
        .linktree-container {
            width: 100%;
            max-width: 600px;
            padding: 2rem;
            border-radius: 1rem;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
            text-align: center;
            <?php if ($linktree_bg_type === 'solid'): ?>
                background-color: <?php echo $linktree_bg_value; ?>;
            <?php elseif ($linktree_bg_type === 'gradient'): ?>
                background: <?php echo $linktree_bg_value; ?>;
            <?php elseif ($linktree_bg_type === 'image'): ?>
                background-image: <?php echo $linktree_bg_value; ?>;
                background-size: cover;
                background-position: center;
            <?php endif; ?>
        }
        .link-item {
            display: flex;
            align-items: center;
            justify-content: center;
            background-color: rgba(255, 255, 255, 0.1);
            border-radius: 0.75rem;
            padding: 1rem 1.5rem;
            margin-bottom: 1rem;
            transition: all 0.2s ease;
            text-decoration: none;
            color: <?php echo $text_color; ?>;
            font-weight: 600;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            position: relative; /* Für das Schloss-Icon */
        }
        .link-item:hover {
            background-color: rgba(255, 255, 255, 0.2);
            transform: translateY(-2px);
            box-shadow: 0 6px 8px rgba(0, 0, 0, 0.15);
        }
        .link-item img {
            width: 2rem;
            height: 2rem;
            border-radius: 0.5rem;
            margin-right: 1rem;
        }
        .category-tabs {
            display: flex;
            justify-content: center;
            margin-bottom: 1.5rem;
            flex-wrap: wrap;
        }
        .category-tab-button {
            padding: 0.75rem 1.5rem;
            border-radius: 0.5rem;
            font-weight: 600;
            color: <?php echo $text_color; ?>;
            background-color: rgba(255, 255, 255, 0.1);
            margin: 0.5rem;
            cursor: pointer;
            transition: all 0.2s ease;
            display: flex; /* Für Icon und Text nebeneinander */
            align-items: center;
        }
        .category-tab-button.active {
            background-color: rgba(255, 255, 255, 0.3);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        .category-tab-button:hover {
            background-color: rgba(255, 255, 255, 0.2);
        }
        .category-content {
            <?php if ($category_bg_type === 'solid'): ?>
                background-color: <?php echo $category_bg_value; ?>;
            <?php elseif ($category_bg_type === 'gradient'): ?>
                background: <?php echo $category_bg_value; ?>;
            <?php elseif ($category_bg_type === 'image'): ?>
                background-image: <?php echo $category_bg_value; ?>;
                background-size: cover;
                background-position: center;
            <?php endif; ?>
            padding: 1.5rem;
            border-radius: 0.75rem;
            margin-bottom: 1.5rem;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        .modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.7);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 1000;
        }
        .modal-content {
            background-color: #ffffff;
            padding: 2rem;
            border-radius: 0.75rem;
            box-shadow: 0 10px 15px rgba(0, 0, 0, 0.2);
            text-align: center;
            color: #333;
        }
        .modal-content input {
            padding: 0.75rem;
            border: 1px solid #cbd5e0;
            border-radius: 0.375rem;
            margin-bottom: 1rem;
            width: 100%;
            box-sizing: border-box;
        }
        .modal-content button {
            padding: 0.75rem 1.5rem;
            background-color: #4c51bf;
            color: white;
            border: none;
            border-radius: 0.375rem;
            cursor: pointer;
            transition: background-color 0.3s ease;
        }
        .modal-content button:hover {
            background-color: #434190;
        }
        .preview-iframe-container {
            width: 100%;
            height: 400px; /* Adjust as needed */
            border: 1px solid #ccc;
            border-radius: 0.75rem;
            overflow: hidden;
            margin-top: 1rem;
        }
        .preview-iframe {
            width: 100%;
            height: 100%;
            border: none;
        }
        .category-dropdown {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #cbd5e0;
            border-radius: 0.375rem;
            margin-bottom: 1rem;
            background-color: rgba(255, 255, 255, 0.1);
            color: <?php echo $text_color; ?>;
            font-weight: 600;
            box-sizing: border-box;
            -webkit-appearance: none; /* Remove default arrow on select */
            -moz-appearance: none;
            appearance: none;
            background-image: url('data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%22292.4%22%20height%3D%22292.4%22%20viewBox%3D%220%200%20292.4%20292.4%22%3E%3Cpath%20fill%3D%22%23<?php echo str_replace('#', '', $text_color); ?>%22%20d%3D%22M287%2069.4a17.6%2017.6%200%200%200-13-5.4H18.4c-5%200-9.3%201.8-12.9%205.4-3.6%203.6-5.4%207.9-5.4%2012.9s1.8%209.3%205.4%2012.9l128%20128c3.6%203.6%207.9%205.4%2012.9%205.4s9.3-1.8%2012.9-5.4l128-128c3.6-3.6%205.4-7.9%205.4-12.9s-1.8-9.3-5.4-12.9z%22%2F%3E%3C%2Fsvg%3E');
            background-repeat: no-repeat;
            background-position: right 0.75rem center;
            background-size: 0.8em auto;
        }
        .password-lock-icon {
            margin-left: 0.5rem;
            color: rgba(255, 255, 255, 0.7); /* Leicht transparentes Weiß */
        }
        .password-error-message {
            color: #ef4444; /* Rot */
            margin-top: -0.5rem;
            margin-bottom: 1rem;
            font-size: 0.9rem;
        }
    </style>
</head>
<body>
    <div class="linktree-container">
        <?php if (!empty($logo_path)): ?>
            <img src="<?php echo $logo_path; ?>" alt="Logo" class="w-24 h-24 mx-auto mb-4 rounded-full shadow-lg">
        <?php endif; ?>
        <h1 class="text-4xl font-bold mb-6 flex items-center justify-center">
            <?php if (!empty($linktree_icon)): ?>
                <i class="<?php echo htmlspecialchars($linktree_icon); ?> text-4xl mr-3"></i>
            <?php endif; ?>
            <?php echo htmlspecialchars($linktree_name); ?>
        </h1>

        <?php if ($category_display === 'tabs' && !empty($categorized_content)): ?>
            <div class="category-tabs">
                <?php $first_category_tab = true; ?>
                <?php foreach ($categorized_content as $cat_id => $content): ?>
                    <button class="category-tab-button <?php echo ($first_category_tab ? 'active' : ''); ?>" onclick="showCategory('<?php echo htmlspecialchars($cat_id); ?>')">
                        <?php if (!empty($content['icon'])): ?>
                            <i class="<?php echo htmlspecialchars($content['icon']); ?> mr-2"></i>
                        <?php endif; ?>
                        <?php echo htmlspecialchars($content['name']); ?>
                    </button>
                    <?php $first_category_tab = false; ?>
                <?php endforeach; ?>
            </div>
        <?php elseif ($category_display === 'dropdown' && !empty($categorized_content)): ?>
            <div class="mb-6">
                <select id="categoryFilterDropdown" class="category-dropdown" onchange="filterCategoryByDropdown()">
                    <?php $first_option = true; ?>
                    <?php foreach ($categorized_content as $cat_id => $content): ?>
                        <option value="<?php echo htmlspecialchars($cat_id); ?>" <?php echo ($first_option ? 'selected' : ''); ?>>
                            <?php echo htmlspecialchars($content['name']); ?>
                        </option>
                        <?php $first_option = false; ?>
                    <?php endforeach; ?>
                </select>
            </div>
        <?php endif; ?>

        <?php $first_category_content = true; ?>
        <?php foreach ($categorized_content as $cat_id => $content): ?>
            <div id="category-<?php echo htmlspecialchars($cat_id); ?>" class="category-content <?php echo (($category_display === 'tabs' || $category_display === 'dropdown') && !$first_category_content ? 'hidden' : ''); ?>" data-category-id="<?php echo htmlspecialchars($cat_id); ?>">
                <h2 class="text-2xl font-semibold mb-4 flex items-center justify-center">
                    <?php if (!empty($content['icon'])): ?>
                        <i class="<?php echo htmlspecialchars($content['icon']); ?> mr-2"></i>
                    <?php endif; ?>
                    <?php echo htmlspecialchars($content['name']); ?>
                </h2>

                <?php foreach ($content['texts'] as $text): ?>
                    <div class="text-block mb-4 p-3 rounded-md bg-gray-700 bg-opacity-30">
                        <p><?php echo nl2br(htmlspecialchars($text['content'])); ?></p>
                    </div>
                <?php endforeach; ?>

                <?php foreach ($content['links'] as $link): ?>
                    <a href="#" class="link-item" data-link-id="<?php echo htmlspecialchars($link['id']); ?>"
                       data-link-url="<?php echo htmlspecialchars($link['url']); ?>"
                       data-password-protected="<?php echo $link['password_protected'] ? 'true' : 'false'; ?>"
                       data-preview-enabled="<?php echo $link['preview_enabled'] ? 'true' : 'false'; ?>"
                       onclick="handleLinkClick(event, this)">
                        <?php if (!empty($link['logo']) && file_exists($upload_dir . $link['logo'])): ?>
                            <img src="<?php echo $upload_dir . htmlspecialchars($link['logo']); ?>" alt="Link Logo">
                        <?php else: ?>
                            <!-- Placeholder image if no logo is uploaded -->
                            <img src="https://placehold.co/32x32/cccccc/333333?text=Link" alt="Link Logo Placeholder">
                        <?php endif; ?>
                        <span><?php echo htmlspecialchars($link['name']); ?></span>
                        <?php if ($link['password_protected']): ?>
                            <i class="fa-solid fa-lock password-lock-icon" id="lock-<?php echo htmlspecialchars($link['id']); ?>"></i>
                        <?php endif; ?>
                        <?php if (!empty($link['description'])): ?>
                            <p class="text-sm opacity-80 mt-1"><?php echo htmlspecialchars($link['description']); ?></p>
                        <?php endif; ?>
                    </a>
                <?php endforeach; ?>
            </div>
            <?php $first_category_content = false; ?>
        <?php endforeach; ?>
    </div>

    <!-- Modals für Passwort und Vorschau -->
    <div id="passwordModal" class="modal hidden">
        <div class="modal-content">
            <h2 class="text-xl font-bold mb-4">Passwort erforderlich</h2>
            <p class="mb-4">Bitte geben Sie das Passwort für diesen Link ein:</p>
            <input type="password" id="passwordInput" placeholder="Passwort eingeben">
            <div id="passwordErrorMessage" class="password-error-message hidden"></div>
            <button onclick="verifyPassword()">Bestätigen</button>
            <button onclick="closeModal('passwordModal')" class="ml-2 bg-gray-500 hover:bg-gray-700">Abbrechen</button>
        </div>
    </div>

    <div id="previewModal" class="modal hidden">
        <div class="modal-content w-11/12 max-w-3xl">
            <h2 class="text-xl font-bold mb-4">Link Vorschau</h2>
            <div class="preview-iframe-container">
                <iframe id="previewIframe" class="preview-iframe" sandbox="allow-scripts allow-same-origin allow-popups allow-forms"></iframe>
            </div>
            <button onclick="closeModal('previewModal')" class="mt-4">Schließen</button>
        </div>
    </div>

    <script>
        let currentLinkElement = null; // Speichert das aktuell angeklickte Link-Element
        // Speichert die IDs der freigeschalteten Links im sessionStorage
        let unlockedLinks = JSON.parse(sessionStorage.getItem('unlockedLinks')) || [];

        // Funktion zum Anzeigen/Ausblenden von Kategorien (für Tabs)
        function showCategory(categoryId) {
            const contents = document.querySelectorAll('.category-content');
            contents.forEach(content => content.classList.add('hidden'));
            const targetContent = document.getElementById('category-' + categoryId);
            if (targetContent) {
                targetContent.classList.remove('hidden');
            }

            const buttons = document.querySelectorAll('.category-tab-button');
            buttons.forEach(button => button.classList.remove('active'));
            const activeButton = document.querySelector(`.category-tab-button[onclick="showCategory('${categoryId}')"]`);
            if (activeButton) {
                activeButton.classList.add('active');
            }
        }

        // Funktion zum Filtern von Kategorien über das Dropdown
        function filterCategoryByDropdown() {
            const selectedCategoryId = document.getElementById('categoryFilterDropdown').value;
            const allCategoryContents = document.querySelectorAll('.category-content');

            allCategoryContents.forEach(content => {
                if (content.dataset.categoryId === selectedCategoryId) {
                    content.classList.remove('hidden');
                } else {
                    content.classList.add('hidden');
                }
            });
        }


        // Funktion zum Öffnen eines Modals
        function openModal(modalId) {
            document.getElementById(modalId).classList.remove('hidden');
            // Fehlermeldung beim Öffnen des Modals zurücksetzen
            const errorMessageDiv = document.getElementById('passwordErrorMessage');
            if (errorMessageDiv) {
                errorMessageDiv.classList.add('hidden');
                errorMessageDiv.textContent = '';
            }
        }

        // Funktion zum Schließen eines Modals
        function closeModal(modalId) {
            document.getElementById(modalId).classList.add('hidden');
            if (modalId === 'passwordModal') {
                document.getElementById('passwordInput').value = ''; // Passwortfeld leeren
                const errorMessageDiv = document.getElementById('passwordErrorMessage');
                if (errorMessageDiv) {
                    errorMessageDiv.classList.add('hidden');
                    errorMessageDiv.textContent = '';
                }
            } else if (modalId === 'previewModal') {
                document.getElementById('previewIframe').src = 'about:blank'; // iFrame leeren
            }
            currentLinkElement = null;
        }

        // Funktion zum Überprüfen des Passworts (serverseitig)
        async function verifyPassword() {
            console.log('verifyPassword function started.');
            const errorMessageDiv = document.getElementById('passwordErrorMessage');
            errorMessageDiv.classList.add('hidden'); // Hide previous error message

            if (!currentLinkElement) {
                console.error('verifyPassword: currentLinkElement is null or undefined.');
                errorMessageDiv.textContent = 'Ein interner Fehler ist aufgetreten. Bitte versuchen Sie es erneut.';
                errorMessageDiv.classList.remove('hidden');
                return;
            }

            const inputPassword = document.getElementById('passwordInput').value;
            const linkId = currentLinkElement.dataset.linkId; // Get the link ID

            console.log('--- Passwortprüfung Debugging (Frontend) ---');
            console.log('Eingegebenes Passwort:', inputPassword);
            console.log('Link ID:', linkId);

            try {
                const response = await fetch('verify_link_password.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({ password: inputPassword, link_id: linkId }),
                });

                const result = await response.json();
                console.log('Server response:', result);

                if (result.success) {
                    // Link erfolgreich freigeschaltet
                    unlockedLinks.push(linkId);
                    sessionStorage.setItem('unlockedLinks', JSON.stringify(unlockedLinks));
                    console.log(`Link ID ${linkId} wurde freigeschaltet und in sessionStorage gespeichert.`);

                    // Optional: Schloss-Icon entfernen oder ändern
                    const lockIcon = document.getElementById(`lock-${linkId}`);
                    if (lockIcon) {
                        lockIcon.classList.remove('fa-lock');
                        lockIcon.classList.add('fa-lock-open'); // Oder ganz entfernen: lockIcon.remove();
                        lockIcon.style.color = 'rgba(0, 255, 0, 0.7)'; // Grün für freigeschaltet
                    }

                    closeModal('passwordModal');
                    if (currentLinkElement.dataset.previewEnabled === 'true') {
                        openPreview(currentLinkElement.dataset.linkUrl);
                    } else {
                        window.open(currentLinkElement.dataset.linkUrl, '_blank');
                    }
                } else {
                    errorMessageDiv.textContent = result.message || 'Falsches Passwort!';
                    errorMessageDiv.classList.remove('hidden');
                }
            } catch (error) {
                console.error('Error during password verification fetch:', error);
                errorMessageDiv.textContent = 'Ein Fehler ist bei der Passwortprüfung aufgetreten. (Netzwerk oder Serverproblem)';
                errorMessageDiv.classList.remove('hidden');
            }
        }

        // Funktion zum Öffnen der Vorschau
        function openPreview(url) {
            const iframe = document.getElementById('previewIframe');
            iframe.src = url;
            openModal('previewModal');
        }

        // Handler für Link-Klicks
        function handleLinkClick(event, linkElement) {
            event.preventDefault(); // Standardverhalten des Links verhindern
            currentLinkElement = linkElement;
            const linkId = linkElement.dataset.linkId;

            console.log('handleLinkClick: Link clicked. Link ID:', linkId);
            console.log('handleLinkClick: Is password protected?', linkElement.dataset.passwordProtected);
            console.log('handleLinkClick: Unlocked links in sessionStorage:', unlockedLinks);

            // Prüfen, ob der Link passwortgeschützt ist UND bereits freigeschaltet wurde
            if (linkElement.dataset.passwordProtected === 'true' && unlockedLinks.includes(linkId)) {
                console.log(`handleLinkClick: Link ID ${linkId} is protected but already unlocked. Opening directly.`);
                if (linkElement.dataset.previewEnabled === 'true') {
                    openPreview(linkElement.dataset.linkUrl);
                } else {
                    window.open(linkElement.dataset.linkUrl, '_blank');
                }
            } else if (linkElement.dataset.passwordProtected === 'true') {
                console.log('handleLinkClick: Link is password protected and not yet unlocked. Opening password modal.');
                openModal('passwordModal');
            } else if (linkElement.dataset.previewEnabled === 'true') {
                console.log('handleLinkClick: Link has preview enabled. Opening preview.');
                openPreview(linkElement.dataset.linkUrl);
            } else {
                console.log('handleLinkClick: Opening link directly (not protected, no preview).');
                window.open(linkElement.dataset.linkUrl, '_blank');
            }
        }

        // Initialen Tab/Dropdown beim Laden der Seite anzeigen/filtern
        document.addEventListener('DOMContentLoaded', () => {
            console.log('DOMContentLoaded: Page fully loaded.');
            const categoryDisplay = "<?php echo $category_display; ?>";
            
            if (categoryDisplay === "tabs") {
                console.log('DOMContentLoaded: Category display is tabs, activating first tab.');
                const firstTabButton = document.querySelector('.category-tab-button');
                if (firstTabButton) {
                    firstTabButton.click(); // Klicke auf den ersten Tab
                }
            } else if (categoryDisplay === "dropdown") {
                console.log('DOMContentLoaded: Category display is dropdown, applying initial filter.');
                const categoryDropdown = document.getElementById('categoryFilterDropdown');
                if (categoryDropdown) {
                    // Trigger change to show the initially selected option
                    categoryDropdown.dispatchEvent(new Event('change'));
                }
            } else { // stacked
                console.log('DOMContentLoaded: Category display is stacked, ensuring all content is visible.');
                // Für stacked ist keine spezielle JS-Aktion erforderlich, da der PHP-Code bereits alles rendert.
            }

            // Beim Laden der Seite den Zustand der Schloss-Icons aktualisieren
            unlockedLinks.forEach(linkId => {
                const lockIcon = document.getElementById(`lock-${linkId}`);
                if (lockIcon) {
                    lockIcon.classList.remove('fa-lock');
                    lockIcon.classList.add('fa-lock-open');
                    lockIcon.style.color = 'rgba(0, 255, 0, 0.7)'; // Grün für freigeschaltet
                }
            });
            console.log('DOMContentLoaded: Initial unlocked links state applied.');
        });
    </script>
</body>
</html>
